<?php
/**
 * FastBroadcastCalendarEvent
 * 
 * @package		FastBroadcast Calendar
 * @version		1.0.0
 * @copyright	Copyright (C) 2012 Horst Noreick, Inc. All rights reserved.
 * @license		GNU/GPLv3 http://www.gnu.org/licenses/gpl-3.0.html
 */

/**
 * This class is important to access the data from the event array given from
 * the API-Function GetEventList and GetEvent.  
 */
class FastBroadcastCalendarEvent
{
	/**
	 * some params..
	 */
	protected $_event 	= null;
	
	/**
	 * person
	 */
	protected $_person 	= null;

	/**
	 * The start time in UNIX time format
	 */
	public $start 		= null;

	/**
	 * The start time in UNIX time format
	 */
	public $end 		= null;	 
	
		
	
	
	/**
	 * Time Conversion the calendar stringtime YYYY-MM-DD HH:MM:SS
	 * is convted to UNIX time value 
	 * 
	 * @param	string 	time
	 * 
	 * @return 	int		UNIX time
	 */
	public function CalTimeToTime( $cal_time )
	{
		list( $year, $month, $day, $hour, $minute, $second) 
			= sscanf( $cal_time, '%4d-%2d-%2d %2d:%2d:%2d');
			
		return @mktime($hour,$minute, $second, $month, $day, $year );
	}

	/**
	 * Init function for this object. It is recommended that each application
	 * shoud use objetc of class FastBroadcastCalendarEvent to access the values
	 * of an event returned by the the FastBroadcast API
	 * 
	 * @param	array	event
	 */
	public function SetEvent( $event )
	{
		if ( is_array( $event ) )
		{
			$this->_event = $event;
			
			$this->start 	= $this->CalTimeToTime( $event['start'] );
			$this->end 		= $this->CalTimeToTime( $event['end'] );
		}
		else
		{
			$this->_event 	= null;
			$this->start	= null;
			$this->end 		= null;
		}		
	}

	/**
	 * Checks wether there is an event assigned 
	 * 
	 * @return bool 
	 */
	public function IsValid()
	{
		return (is_array( $this->_event));
	}

	/**
	 * Checks wether the Event-start matches the current day 
	 * 
	 * @return bool 
	 */
	public function IsToday()
	{
		return (date('Ymd') == date('Ymd', $this->start ));
	}
	
	/**
	 * Compares the Event-Start with a UNIX Timestamp 
	 * 
	 * @param int 		time
	 * @param string	if precision is day, the hour, minute and second
	 * 					values are ignored 
	 * 
	 * @return bool 	returns true on equal
	 */
	public function CompareStart( $time, $precision = 'day' )
	{
		$result = 0;
		switch ( $precision )
		{
		case 'day':
			$result = (date('Ymd', $time ) == date('Ymd', $this->start ));	
			break;
		default:
			$result = (date('YmdHi', $time ) == date('YmdHi', $this->start ));
		}
		return $result;
	}
	
	
	/**
	 * This Method will return an Object of NosFastBroadcastPerson 
	 * 
	 * @return mixed 	if there is an Object the method returns an object of
	 * 					the class NosFastBroadcastPerson. otherwise it returns
	 * 					null
	 */
	protected function Person()
	{
		if ( is_null($this->_person) )
			$this->_person = new NosFastBroadcastPerson();
		return $this->_person;			
	}

	/**
	 * Get the summary of event-struct.
	 *
	 * @return  string	the Title of the Event	
	 */	
	public function Summary()
	{
		return $this->_event['summary']; 
	}	

	/**
	 * Get the eventid of event-struct.
	 *
	 * @return  string	the eventid	
	 */	
	public function Id()
	{
		return $this->_event['eventid']; 
	}
		
	/**
	 * Get the eventnr of event-struct.
	 *
	 * @return  string	the eventnr	
	 */	
	public function Nr()
	{
		return $this->_event['eventnr']; 
	}	

	/**
	 * Get a formatted string of the events creation date
	 *
	 * @param   string 	format: see PHP date function 
	 *
	 * @return  string	the formatted creation date	
	 */	
	public function Created( $format )
	{
		$created = $this->CalTimeToTime( $this->_event['created'] );
		return @date( $format, $created );
	}	
	
	/**
	 * The Name of the user who created the Event
	 * 
	 * @return string username
	 */
	public function CreatedBy()
	{
		return $this->_event['created_by']; 
	}	
	
	/**
	 * Get a formatted string of the events last modified date
	 *
	 * @param   string 	format: see PHP date function 
	 *
	 * @return  string	the formatted modified date	
	 */	
	public function Modified( $format )
	{
		$modified = $this->CalTimeToTime( $this->_event['modified'] );
		return @date( $format, $modified );
	}	
	
	
	/**
	 * The Name of the user who last modified the Event
	 * 
	 * @return string username
	 */
	public function ModifiedBy()
	{
		return $this->_event['modified_by']; 
	}	
	
	
	/**
	 * Is the hitcounter part of the event-array?
	 * 
	 * @return bool 
	 */
	public function IsHits()
	{
		return (isset($this->_event['hits']));
	}
	/**
	 * The current state of the hitcounter
	 * 
	 * @return int hits
	 */
	public function Hits()
	{
		return ($this->IsHits())? $this->_event['hits'] : 0; 
	}	
	
	
	
	
	/**
	 * Is the clientstate part of the event-array?
	 * 
	 * @return bool 
	 */
	public function IsClientState()
	{
		return (isset($this->_event['clientstate']));
	}

	/**
	 * Gives back the Clientstate if it exists in the event-array
	 * 
	 * @return int 	clientstate 
	 * 				0 = the event does not belong to a client
	 * 				1 or higher = the location or the organizer is 
	 * 							registered as a client 
	 */
	public function ClientState()
	{
		return ($this->IsClientState())? $this->_event['clientstate'] : 0; 
	}	
	
	/**
	 * Is there a scene in the event-array?
	 * 
	 * @return bool 	 
	 */
	public function IsScene()
	{
		return (isset($this->_event['scene']));
	}
	
	/**
	 * Gives back the Scene of the event
	 * 
	 * @return string 	e.g. 'salsa', 'tango', 'mediation'... 	 
	 */
	public function Scene()
	{
		return ($this->IsScene())? $this->_event['scene'] : '';		 
	}

	
	/**
	 * Is there a category in the event-array?
	 * 
	 * @return bool 	 
	 */
	public function IsCategory()
	{
		return (isset($this->_event['category']));
	}

	/**
	 * Gives back the Category of the event
	 * 
	 * @return string 	e.g. 'party', 'concert', 'retreat'... 	 
	 */
	public function Category()
	{
		return ($this->IsCategory())? $this->_event['category'] : '';
	}	

	/**
	 * Is there a category-group in the event-array?
	 * 
	 * @return bool 	 
	 */
	public function IsCatgroup()
	{
		return (isset($this->_event['catgroup']));
	}
	
	/**
	 * Gives back the Category-group of the event
	 * 
	 * @return string 	one of each 'event', 'workshop', 'lesson' 	 
	 */
	public function Catgroup()
	{
		return ($this->IsCatgroup())? $this->_event['catgroup'] : '';		 
	}	
	
	/**
	 * Is there a start-value in the event-array?
	 * 
	 * @return bool 	 
	 */
	public function IsStart()
	{
		return (isset($this->_event['start']));
	}
	
	/**
	 * Get a formatted string of the events start date
	 *
	 * @param   string 	format: see PHP date function 
	 *
	 * @return  string	the formatted start date	
	 */	
	public function Start( $format )
	{
		return @date( $format, $this->start );
	}
	
	/**
	 * Is there an end-value in the event-array?
	 * 
	 * @return bool 	 
	 */
	public function IsEnd()
	{
		return (isset($this->_event['end']));
	}
	
	/**
	 * Get a formatted string of the events end date
	 *
	 * @param   string 	format: see PHP date function 
	 *
	 * @return  string	the formatted end date	
	 */	
	public function End( $format )
	{
		return @date( $format, $this->end );
	}
	

	/**
	 * Is the requested location number valid?
	 * 
	 * @param 	int index of the location
	 * 
	 * @return bool 	 
	 */
	public function IsLocation( $index = 0 )
	{
		return (isset($this->_event['personlist']['location'][$index]));
	}
	

	/**
	 * Get an FastBroadcastPerson object of the Location
	 * 
	 * @param 	int 	index of the location. The default is zero.
	 * 
	 * @return 	mixed  	gives back an object of the class FastBroadcastPerson
	 * 					when the Location exists; otherwise null
	 */
	public function Location( $index = 0 )
	{
		$person = null;
		if ( $this->IsLocation( $index ) )
		{
			require_once dirname(__FILE__).'/FastBroadcastPerson.class.php';		
			$person = new FastBroadcastPerson( 
				$this->_event['personlist']['location'][$index] 
			);
		}
		return $person; 
	}

	/**
	 * Is the requested organizer valid?
	 * 
	 * @param 	int index of the organizer
	 * 
	 * @return bool 	 
	 */
	public function IsOrganizer( $index = 0 )
	{
		return (isset($this->_event['personlist']['organizer'][$index]));
	}
	
	/**
	 * Get an FastBroadcastPerson object of the Organizer
	 * 
	 * @param 	int 	index of the organizer, The default is zero.
	 * 
	 * @return 	mixed  	gives back an object of the class FastBroadcastPerson
	 * 					when the Location exists; otherwise null
	 */
	public function Organizer( $index = 0 )
	{
		$person = null;
		if ( $this->IsOrganizer( $index ) )
		{
			require_once dirname(__FILE__).'/FastBroadcastPerson.class.php';			
			$person = new FastBroadcastPerson( 
				$this->_event['personlist']['organizer'][$index] 
			);
		}
		return $person; 
	}

	/**
	 * Is the requested dj valid?
	 * 
	 * @param 	int index of the dj
	 * 
	 * @return bool 	 
	 */
	public function IsDj( $index = 0 )
	{
		return (isset($this->_event['personlist']['dj'][$index]));
	}
	
	/**
	 * Get an FastBroadcastPerson object of the Dj 
	 * 
	 * @param 	int 	index of the dj, The default is zero.
	 * 
	 * @return 	mixed  	gives back an object of the class FastBroadcastPerson
	 * 					when the Location exists; otherwise null
	 */
	public function Dj( $index = 0 )
	{
		$person = null;
		if ( $this->IsDj( $index ) )
		{
			require_once dirname(__FILE__).'/FastBroadcastPerson.class.php';			
			$person = new FastBroadcastPerson( $this->_event['personlist']['dj'][$index] );
		}
		return $person; 
	}
	
	/**
	 * If the indexed Image exists this function will return the name of the image
	 * 
	 * @param	int	index
	 * 
	 * @return	bool
	 */	
	function ImageName( $index = 0 )
	{
		return ( $this->IsImage($index ))? $this->_event['images'][$index]['NAME'] : '';
	}
	
	/**
	 * The URL of Files to Download
	 * 
	 * @param 	int		the index of the file in the event array
	 * 
	 * @return	string	the path, if an valid download-row is indexed,
	 * 					otherwise an empty string is returned 
	 */
	function DownloadURL( $index )
	{
		$url = '';
		if ( isset( $this->_event['downloads'][$index]) )
		{
			$url	= FastBroadcast::$scheme
					. FastBroadcast::HOST
					. FastBroadcast::DOWNLOAD_PATH
					. 'rpc_method=V001_Calendar_GetFile&amp;'
					. 'eventid='. $this->Id(). '&amp;'
					. 'file='. $this->_event['downloads'][$index]['NAME'];
		}
		return $url;
	}
	
	/**
	 * if a flyer exist in the imagetable this function will return true
	 * 
	 * @param	int 	index
	 * 
	 * @return	bool
	 */
	public function IsFlyer( &$index )
	{
		$isFlyer 	= false;
		$index		= 0; 
		if ( isset( $this->_event['images'][0] ) )
		{
			foreach ( $this->_event['images'] as $image )
			{
				if ( $image['NAME'] == 'flyer_01' )
				{
					$isFlyer 	= true;					
					break;
				}
				$index++;
			}
		}
		return $isFlyer;
	}
	
	
	/**
	 * The URL of an image identified by index 
	 * 
	 * @param	int		index
	 * @param	int		width oft the image
	 * @param	int		height of the image
	 * 
	 * @return	string	the url of the image if the index is valid
	 * 					otherwise an empty string
	 */
	function ImageURL( $index=0, $width = 0, $height = 0  )
	{
		$url = '';
		if ( $this->IsImage( $index ) )
		{
			$url	= FastBroadcast::$scheme
					. FastBroadcast::HOST
					. FastBroadcast::IMAGE_PATH
					. 'rpc_method=V001_Calendar_GetImage&amp;'
					. 'eventid='. $this->Id(). '&amp;'
					. 'file='. $this->ImageName($index);
	
			if ( $width > 0 )
				$url .= "&amp;width=$width";
			if ( $height > 0 )
				$url .= "&amp;height=$height";
		}
		return $url;				
	}
	
	/**
	 * a complete HTML img-tag is created and returned by this method
	 * the params are equal to the method ImageURL
	 * 
	 * @param	int		index
	 * @param	int		width
	 * @param	int		height
	 * 
	 * @return	string	the html img-tag
	 */
	function ImageTag( $index=0, $width = 0, $height = 0 )
	{
		$tag = '';
		$url = $this->ImageURL($index, $width, $height );
		if ( !empty( $url ) )
		{
			$width = ($width)? '"width='.$width.'"' : '';
			$tag = "<img $width src=\"$url\" alt=\"\" />";
		} 
		return $tag; 
	}	
	/**
	 * Checks the existance of an image at pos $index
	 * 
	 * @param	int		index
	 * 
	 * @return 	bool
	 */	
	public function IsImage( $index = 0 )
	{
		return (isset($this->_event['images'][$index]));
	}
		
	/**
	 * Check for existens of a slogan-string in event-array.
	 *
	 * @return  bool	
	 */	
	public function IsSlogan()
	{
		return (isset( $this->_event['slogan']) && $this->_event['slogan']);
	}

	/**
	 * Get the slogan of event-struct.
	 *
	 * @return  string	the slogan or an empty string	
	 */	
	public function Slogan()
	{
		return ($this->IsSlogan())? $this->_event['slogan'] : ''; 
	}	

	/**
	 * Check for existens of a valid cost row in event-array.
	 *
	 * @return  bool	
	 */	
	public function IsCost()
	{
		return (isset( $this->_event['cost']) && $this->_event['cost']);	
	}
	
	
	
	/**
	 * Check wether the event is for free
	 *
	 * @return  bool	
	 */	
	public function IsCostFree()
	{
		return (isset( $this->_event['cost']['FREE']) && $this->_event['cost']['FREE']);	
	}
	
	/**
	 * Check wether the event is for free
	 *
	 * @return  bool	
	 */	
	public function CostFree()
	{
		return ($this->IsCostFree())? $this->_event['cost']['FREE'] : '0';		
	}

	/**
	 * Is there a cost per unit entry in the event?
	 * 
	 * @return	bool
	 */
	public function IsCostPerUnit()
	{
		return (isset( $this->_event['cost']['PERUNIT']) && $this->_event['cost']['PERUNIT']);	
	}
	
	/**
	 * Give back the cost per unit value if it exists
	 * 
	 * @return string	the cost per unit 
	 */
	public function CostPerUnit()
	{
		return ($this->IsCostPerUnit())? $this->_event['cost']['PERUNIT'] : '';		
	}

	/**
	 * Is there a VVK entry in the event? (VVK means Vorverkauf)
	 * 
	 * @return	bool
	 */
	public function IsCostVVK()
	{
		return (isset( $this->_event['cost']['VVK']) && $this->_event['cost']['VVK']);	
	}
	
	/**
	 *	Gives back the VVK (Vorverkauf) value if it exists in the event 
	 * 
	 * @return string
	 */
	public function CostVVK()
	{
		return ($this->IsCostVVK())? $this->_event['cost']['VVK'] : '';		
	}
	
	/**
	 * Is there a AK entry in the event? (AK means Abendkasse)
	 * 
	 * @return	bool
	 */
	public function IsCostAK()
	{
		return (isset( $this->_event['cost']['AK']) && $this->_event['cost']['AK']);	
	}

	/**
	 *	Gives back the AK (Abendkasse) value if it exists in the event 
	 * 
	 * @return string
	 */
	public function CostAK()
	{
		return ($this->IsCostAK())? $this->_event['cost']['AK'] : '';		
	}

	/**
	 * Checks wether the textfield for the price for mor complex prices is used
	 * 
	 * @return	bool
	 */
	public function IsCostText()
	{
		return (isset( $this->_event['cost']['TEXT']) && $this->_event['cost']['TEXT']);	
	}
	/**
	 * Give back the text of the cost
	 * 
	 * @return	string
	 */
	public function CostText()
	{
		return ($this->IsCostText())? $this->_event['cost']['TEXT'] : '';		
	}

	/**
	 * Is the table used for the specification of the event price
	 * 
	 * @return	bool
	 */
	public function IsCostTable()
	{
		return (isset( $this->_event['cost']['costtable']) );	
	}
	
	/**
	 * the cost may be specified in a two column table
	 * 
	 * @return 	mixed	an array for the costtabele if it exists or null
	 */
	public function CostTable()
	{
		return ($this->IsCostTable())? $this->_event['cost']['costtable'] : null;		
	}

	/**
	 * A program may be specified as a table width two columns represented as an array
	 * by the user for an event.
	 * 
	 * @return bool
	 */
	public function IsProgram()
	{
		return (isset( $this->_event['program']) );	
	}
	
	/**
	 * A program may be specified as a table width two columns represented as an array
	 * by the user for an event.
	 * 
	 * @return	mixed	array if there is a program otherwise null
	 */
	public function Program()
	{
		return ($this->IsProgram())? $this->_event['program'] : null;		
	}

	/**
	 * does the event have something to download (php-file for example)
	 * 
	 * @return	bool
	 */
	public function IsDownload()
	{
		return (isset( $this->_event['downloads']) );	
	}
	
	/** 
	 * download are specified in an array with two columns, the text and the 
	 * file specification
	 * 
	 * @return	mixed	array, if there is something to download otherwise null
	 */
	public function Download()
	{
		return ($this->IsDownload())? $this->_event['downloads'] : null;		
	}
	
	/**
	 * this method can be used to resize the amount of text of the event description
	 * 
	 * @param	string	the text to resize
	 * @param	int		max lines oft the resized text
	 * @param	int		max char per line ( requred to braek paragraphs to lines)
	 * @param	string	for html output the eol should be set to a br tag
	 * @param	string	if the result is smaller when the input, an ellipse or an 
	 * 					other string is added to the end of the text
	 */
	function ResizeText( $text_in, $max_lines, $max_chars_per_line, $eol, $ellipse = ' ...' )
	{
		$lines = explode( '\n', $text_in );		
		$text_out = '';	
		
		$i = 0;
		foreach ( $lines as $line )
		{
			$tlines = strlen( $line ) / $max_chars_per_line + 1;
			if ( $i + $tlines >= $max_lines )
			{
				$chars = ($max_lines - $i) * $max_chars_per_line;
				$line = substr( $line, 0, $chars ); 
				$line = substr( $line, 0, strrpos($line,' ') );
				if ( $ellipse )
				 	$line .= $ellipse;
			}
			$text_out .= $line . $eol;
			$i += $tlines;
			if ( $i >= $max_lines )
				break;
		}
		return $text_out;
	}	

	/**
	 * complex formattings can be done with this method. Variables must be enclosed
	 * with brackets {variable}. 
	 * sample: "{location_city}: {start_H}:{start_i} {location_fullname}"
	 * for the resolution of time-variables see the date funktion in php.net 
	 * 
	 * @param	string	format
	 * 
	 * @return	the formatstring where the valid params are resolved
	 */
	public function sprintf( $format )
	{
		$pattern = "/{[^}]*}/";
		preg_match_all($pattern, $format, $matches);
		
		$location = $this->Location();
		$organizer = $this->Location();		
			
		foreach ( $matches[0] as $search )
		{
			$replace 	= '';
			$var		= trim( $search, '{}' );
			$var_parts 	= explode( '_', $var );
			
			if ( count( $var_parts ) > 1 )
			{
				switch ( $var_parts[0] )
				{
				case 'location':
					if ( is_object( $location ) )
						$replace = $location->get( $var_parts[1] );
					break;
				case 'organizer':
					if ( is_object( $organizer ) )
						$replace = $organizer->get( $var_parts[1] );
					break;
				case 'start':
					switch ( $var_parts[1] )
					{
					case 'D':
					case 'l':
					case 'F':
						$replace = FastBroadcast::Translate( $this->Start( $var_parts[1] ) );
						break;
						//else weiter
					default:
						$replace = $this->Start( $var_parts[1] );
					}
					break;
				case 'end':
					$replace = $this->End( $var_parts[1] );
					break;
				}
			}
			else
			{
//				switch ( $var )
//				{
//				}
			}
			$format = str_replace( $search, $replace, $format );			
		}
		
		return $format;
	}
	
	
	/**
	 * Check for existens of a event-Description
	 *
	 * @return  bool	
	 */	
	public function IsDescription()
	{
		return (isset( $this->_event['description']) && $this->_event['description'])? true : false;
	}
	
	/**
	 * Get the description of event-struct. If the values for lines and columns are 
	 * specified the method will use the ResiseTest Method to resize the text
	 *
	 * @param   int		columns for resizing
	 * @param   int		lines for resizing*
	 * @param	string	the linebreakes usually will be replaced with a br tag
	 *
	 * @return  string	the description or an empty string resized by the params	
	 */	
	public function Description( $lines = 0, $columns = 0, $eol = '<br />')
	{
		$description = '';
		if ( $this->IsDescription() )
		{
			if ( $columns && $lines )
				$description = $this->ResizeText( $this->_event['description'], $lines, $columns, $eol );
			else 
				$description = str_replace( "\\n", $eol, $this->_event['description'] );
		}
		return $description;
	}
	
}
