<?php
/**
 * This class acts as client and encapsulates the requests to the FastBroadcast 
 * Server
 * 
 * @package		FastBroadcast Client
 * @version		1.0.0
 * @copyright	Copyright (C) 2012 Horst Noreick, Inc. All rights reserved.
 * @license		GNU/GPLv3 http://www.gnu.org/licenses/gpl-3.0.html
 */

/**
 * class FastBroadcastHttpClient
 * 
 * Objects of this should be received via the FastBroadcast class
 * sample: 	$httpClient = FastBroadcast::HttpClient();
 */
class FastBroadcastHttpClient
{
	/**
	 *  member to to build the url
	 */
	private $response = null;
	
	/**
	 * The Constructor 
	 */
	private function __construct( )
	{
//		date_default_timezone_set('Europe/Berlin');			
	}
	
	/**
	 * singleton! nothing to clone
	 */
	private function __clone()
	{
	
	}
	
	/**
	 * Creating an object of FastBroadcastHttpClient
	 * 
	 * @return	object	the one and only instance of this class
	 */
	static public function GetInstance()
	{
		static $instance = null;
		if ( is_null( $instance ) )
			$instance = new FastBroadcastHttpClient();
		
		return $instance;  
	} 
	
	/**
	 * Call for API-Method VOO1_Calendar_GetEvent
	 * 
	 * @param	array	see API-Documentation for this method
	 * 					on http://www.fastbroadcast.de
	 * 
	 * @return	int		retcode: FastBroadcast::SUCCESS or an error nr 
	 */
	public function GetEvent( array $params )
	{
		$params['rpc_method'] = 'V001_Calendar_GetEvent';
		return $this->RemoteRequest( $params );		
	}		
	
	/**
	 * Call for API-Method VOO1_Calendar_GetEventList
	 * 
	 * @param	array	see API-Documentation for this method
	 * 					on http://www.fastbroadcast.de
	 * 
	 * @return	int		retcode: FastBroadcast::SUCCESS or an error nr 
	 */
	public function GetEventList( $params )
	{
		$params['rpc_method'] = 'V001_Calendar_GetEventList';
		return $this->RemoteRequest( $params );		
	}		

	
	/**
	 * Access to FastBroadcast with CURL
	 * 
	 * @param 	string	the query
	 * 
	 * @return 	string	the data from the response
	 */
	protected function RequestWith_CURL( $query )
	{
		// build the full url
		$url 	= FastBroadcast::$scheme 
				. FastBroadcast::HOST 
				. FastBroadcast::RPC_PATH; 
				
	
		// curl-Request: POST
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $url );
		curl_setopt($ch, CURLOPT_HTTPHEADER, array("Accept: application/json") );				
		curl_setopt($ch, CURLOPT_AUTOREFERER, true);
		curl_setopt($ch, CURLOPT_VERBOSE, 1);
		curl_setopt($ch, CURLOPT_POST, true );
		curl_setopt($ch, CURLOPT_POSTFIELDS, $query );		
		curl_setopt($ch, CURLOPT_RETURNTRANSFER,1);
		curl_setopt($ch, CURLOPT_USERAGENT, __CLASS__ . '_' . FastBroadcast::VERSION );
		curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, FastBroadcast::TIMEOUT );
		
		$jresponse = curl_exec($ch);
		curl_close ($ch);
		return $jresponse;
	}
	
	
	
	/**
	 * function by Buoysel found at php.net
	 * 
	 * @param	string	the raw data given from the server
	 * 
	 * @return	string	the unchunked data
	 */
	function UnchunkResult( $data ) 
	{
	    $fp 		= 0;
	    $outData 	= "";
	    while( $fp < strlen( $data ) ) 
	    {
	        $rawnum 	= 	substr( $data, $fp, strpos( substr( $data, $fp ), "\r\n" ) + 2 );
	        $num 		= 	hexdec( trim( $rawnum ) );
	        $fp 		+= 	strlen( $rawnum );
	        $chunk 		= 	substr( $data, $fp, $num );
	        $outData 	.= $chunk;
	        $fp 		+= strlen( $chunk );
	    }
	    return $outData;
	}	
	
	/**
	 * Access to FastBroadcast with fsockopen
	 * 
	 * @param 	string	the query
	 * 
	 * @return 	string	the data from the response
	 */
	protected function RequestWith_fsockopen( $query )
	{
		$jresponse = '';
	
		$fp = @fsockopen( FastBroadcast::HOST, 80, $errno, $errst, FastBroadcast::TIMEOUT );
		if ( $fp )
		{
			$request	= 	"POST " . FastBroadcast::RPC_PATH . " HTTP/1.1\r\n"
						. 	"Host: " . FastBroadcast::HOST . "\r\n"
						. 	"Accept: application/json\r\n"
						. 	"User-Agent: " . __CLASS__ . '_' . FastBroadcast::VERSION . "\r\n"						
 						.	"Content-Type: application/x-www-form-urlencoded\r\n"
        				.	"Content-Length: " . strlen( $query ) . "\r\n";
        				
			if ( isset($_SERVER['HTTP_REFERER']) )
   				$request .= "Referer: {$_SERVER['HTTP_REFERER']}\r\n"; 
        				
			$request	.=	"\r\n"
						.	$query;
						
   			$response 	= '';   			
		    fwrite( $fp, $request );
    		while ( !feof( $fp ) ) 
        		$response .= fgets( $fp, 4096 );
    		fclose($fp);				
    		
			$data = substr($response, (strpos($response, "\r\n\r\n")+4));
			if (strpos(strtolower($response), "transfer-encoding: chunked") !== FALSE)
    			$data = $this->UnchunkResult($data);    		
    			
    		$jresponse = $data;    		
		}
		return $jresponse;		
	}
	
	/**
	 * Builing the querystring from th given key-value pairs in the params-array
	 * 
	 * @param	array 	the params
	 * 
	 * @return	string	querystring
	 */
	protected function BuildQuery( array $params )
	{
		// Build a querystring from the $params-array
		$query = array();
		foreach ($params as $k => $v )
			$query[] = $k . '='. rawurlencode( $v );	
			
		return implode( '&', $query );
	}	
	
	/**
	 * RemoteRequest
	 * 
	 * @param	array	the params for the API-Methos call
	 * 
	 * @return	int		returncode 	 
	 * 					FastBroadcast::SUCCESS if the request was successfull handled
	 * 					even when the server has no Data to send
	 */
	public function RemoteRequest( array $params )
	{
		if ( FastBroadcast::APPID() )
			$params['appid'] = FastBroadcast::APPID();		
	
		$query = $this->BuildQuery( $params );
	
		// Do the request and receive data from fastbroadcast.de 
		if ( function_exists('curl_init') && function_exists('curl_exec') ) 
			$jresponse = $this->RequestWith_CURL( $query );
		else 
			$jresponse = $this->RequestWith_fsockopen( $query );
		
		// try to convert the response to an array with json
		$response = json_decode( $jresponse, true );
		if ( is_array( $response ) ) 
		{
			$retcode = (isset( $response['retcode'] )) ? 
						$response['retcode'] : FastBroadcast::ERROR;
		}
		else 
		{
			// the server did not give back a valid json string
			$retcode = FastBroadcast::ERROR;
		}
		
		switch ( $retcode )	
		{
			case FastBroadcast::NO_DATA_FOUND:		
			case FastBroadcast::SUCCESS:
				$this->response = $response;
				$retcode	= FastBroadcast::SUCCESS; 
				break;
			default:
				if ( is_array( $response ) ) 
				{
					FastBroadcast::SetError( $retcode, 'Remote Server Error: '
									.$response['error'][0]['error'] 
									. ' ' 
									. $response['error'][0]['message'] );
				}
				else 
				{
					FastBroadcast::SetError( $retcode, 'Remote Server Error: '.$jresponse );
				}				
		}	
		
		return $retcode;	
	}
	
	/**
	 * The result from the remote method call
	 * 
	 * @param	string 	type. By default the type is an empty string and the
	 * 					result array from the response is given.
	 * 
	 * @return	array	the returned array may be specified by the param type
	 */
	public function Result( $type = '' )
	{
		switch ( $type )
		{
		case 'raw':
			return $this->response;
		}
		
		return $this->response['result'];		
	}
}
