<?php
/**
 * FastBroadcastCalendarEvent
 * 
 * @package		FastBroadcast Calendar
 * @version		1.0.0
 * @copyright	Copyright (C) 2012 Horst Noreick, Inc. All rights reserved.
 * @license		GNU/GPLv3 http://www.gnu.org/licenses/gpl-3.0.html
 */

// This Line is required to pass the JEDChecker in Joomla
defined('_JEXEC') or die('Restricted access');
  
  
 
/**
 * This class is important to access the data from the event array given from
 * the API-Function GetEventList and GetEvent.  
 */
class FastBroadcastCalendarEvent
{
	/**
	 * some params..
	 */
	protected $_event 	= null;
	
	/**
	 * person
	 */
	protected $_person 	= null;

	/**
	 * The start time in UNIX time format
	 */
	public $start 		= null;

	/**
	 * The start time in UNIX time format
	 */
	public $end 		= null;	 
	

	/**
	 * person
	 */
	protected $output_format	= 'html';
	

	/**
	 * schema_org
	 */
	protected $_schema_org 		= null;
	
	
	/**
	 * Time Conversion the calendar stringtime YYYY-MM-DD HH:MM:SS
	 * is convted to UNIX time value 
	 * 
	 * @param	string 	time
	 * 
	 * @return 	int		UNIX time
	 */
	public function CalTimeToTime( $cal_time )
	{
		list( $year, $month, $day, $hour, $minute, $second) 
			= sscanf( $cal_time, '%4d-%2d-%2d %2d:%2d:%2d');
			
		return @mktime($hour,$minute, $second, $month, $day, $year );
	}

	/**
	 * Convert
	 * 
	 * @param	string	string to convert
	 */
	public function Convert( $input )
	{
		if ( $this->output_format == 'html' )
			return htmlspecialchars( $input );
		return $input; 
	}
	
	/**
	 * Info
	 * 
	 * @param	string	html-string with information
	 */
	public function Info()
	{
		$htm = 'FastBroadcast IDs :: ';	
			
		$htm .= '# ' . $this->ID() . ' - EventID &nbsp; ';
				
		$location = $this->Location(); 
		if ( $location )
			$htm .= '# ' . $location->ID() . ' - ' . $location->FullName() .  '&nbsp;' ;
		
		$organizer = $this->Organizer(); 
		if ( $organizer )
			$htm .= '# ' . $organizer->ID() . ' - ' . $organizer->FullName() .  '&nbsp;' ;
			
		return $htm;
	
/*	
		$htm = '<b>FastBroadcast IDs</b><br />';	
			
		$htm .= '# ' . $this->ID() . ' - EventID<br />';
				
		$location = $this->Location(); 
		if ( $location )
			$htm .= '# ' . $location->ID() . ' - ' . $location->FullName() . '<br />';
		
		$organizer = $this->Organizer(); 
		if ( $organizer )
			$htm .= '# ' . $organizer->ID() . ' - ' . $organizer->FullName() . '<br />';
			
		return $htm;
*/		
	}
	
	/**
	 * Init function for this object. It is recommended that each application
	 * shoud use objetc of class FastBroadcastCalendarEvent to access the values
	 * of an event returned by the the FastBroadcast API
	 * 
	 * @param	array	event
	 */
	public function SetEvent( $event )
	{
		if ( is_array( $event ) )
		{
			$this->_event = $event;
			
			$this->start 	= (isset($event['start']))? $this->CalTimeToTime( $event['start'] ) : 0;
			$this->end 		= (isset($event['end']))? $this->CalTimeToTime( $event['end'] ) : 0 ;
		}
		else
		{
			$this->_event 	= null;
			$this->start	= null;
			$this->end 		= null;
		}		
	}

	/**
	 * 
	 * @param object microdata
	 */
	public function SetSchemaOrg( NosSemanticWeb $schema_org )
	{
		$this->_schema_org = $schema_org;
	}	

	/**
	 * 
	 * @return object NosSemanticWeb
	 */
	public function SchemaOrg()
	{
		if ( !is_object($this->_schema_org) )
		{
			require_once dirname(__FILE__).'/NosSemanticWeb.class.php';
			$this->_schema_org = new NosSemanticWeb;
		}
		return $this->_schema_org;
	}	

	
	/**
	 * Checks wether there is an event assigned 
	 * 
	 * @return bool 
	 */
	public function IsValid()
	{
		return (is_array( $this->_event));
	}

	/**
	 * Checks wether the Event-start matches the current day 
	 * 
	 * @return bool 
	 */
	public function IsToday()
	{
		return (date('Ymd') == date('Ymd', $this->start ));
	}
	
	/**
	 * Compares the Event-Start with a UNIX Timestamp 
	 * 
	 * @param int 		time
	 * @param string	if precision is day, the hour, minute and second
	 * 					values are ignored 
	 * 
	 * @return bool 	returns true on equal
	 */
	public function CompareStart( $time, $precision = 'day' )
	{
		$result = 0;
		switch ( $precision )
		{
		case 'day':
			$result = (date('Ymd', $time ) == date('Ymd', $this->start ));	
			break;
		default:
			$result = (date('YmdHi', $time ) == date('YmdHi', $this->start ));
		}
		return $result;
	}
	
	
	/**
	 * This Method will return an Object of NosFastBroadcastPerson 
	 * 
	 * @return mixed 	if there is an Object the method returns an object of
	 * 					the class NosFastBroadcastPerson. otherwise it returns
	 * 					null
	 */
	protected function Person()
	{
		if ( is_null($this->_person) )
			$this->_person = new NosFastBroadcastPerson();
		return $this->_person;			
	}

	/**
	 * Get the summary of event-struct.
	 *
	 * @return  string	the Title of the Event	
	 */	
	public function Summary()
	{
		return $this->Convert( $this->_event['summary'] ); 
	}	

	/**
	 * Get the eventid of event-struct.
	 *
	 * @return  string	the eventid	
	 */	
	public function Id()
	{
		return $this->_event['eventid']; 
	}
		
	/**
	 * Get the eventnr of event-struct.
	 *
	 * @return  string	the eventnr	
	 */	
	public function Nr()
	{
		return $this->_event['eventnr']; 
	}	

	/**
	 * Get a formatted string of the events creation date
	 *
	 * @param   string 	format: see PHP date function 
	 *
	 * @return  string	the formatted creation date	
	 */	
	public function Created( $format )
	{
		$created = $this->CalTimeToTime( $this->_event['created'] );
		return @date( $format, $created );
	}	
	
	/**
	 * The Name of the user who created the Event
	 * 
	 * @return string username
	 */
	public function CreatedBy()
	{
		return $this->_event['created_by']; 
	}	
	
	/**
	 * Get a formatted string of the events last modified date
	 *
	 * @param   string 	format: see PHP date function 
	 *
	 * @return  string	the formatted modified date	
	 */	
	public function Modified( $format )
	{
		$modified = $this->CalTimeToTime( $this->_event['modified'] );
		return @date( $format, $modified );
	}	
	
	
	/**
	 * The Name of the user who last modified the Event
	 * 
	 * @return string username
	 */
	public function ModifiedBy()
	{
		return $this->_event['modified_by']; 
	}	


	/**
	 * Facebook ist an array of facbook stuff assigned to this event
	 * 
	 * @return bool true if next exists
	 */
	public function IsFacebook( $category )
	{
		return isset($this->_event['facebook'][$category]); 
	}	
	
	/**
	 * 
	 * 
	 * @return mixed array or null
	 */
	public function Facebook( $category )
	{
		return isset($this->_event['facebook'][$category])? $this->_event['facebook'][$category] : null; 
	}	
	
	/**
	 * Next ist an array of events, this function checks the existance of this array
	 * 
	 * @return bool true if next exists
	 */
	public function IsNext()
	{
		return isset($this->_event['next']); 
	}	
	
	/**
	 * Next ist an array of events
	 * 
	 * @return mixed array or null
	 */
	public function Next()
	{
		return isset($this->_event['next'])? $this->_event['next'] : null; 
	}	
	
	
	/**
	 * Is the hitcounter part of the event-array?
	 * 
	 * @return bool 
	 */
	public function IsHits()
	{
		return (isset($this->_event['hits']));
	}
	/**
	 * The current state of the hitcounter
	 * 
	 * @return int hits
	 */
	public function Hits()
	{
		return ($this->IsHits())? $this->_event['hits'] : 0; 
	}	
	
	
	
	
	/**
	 * Is the clientstate part of the event-array?
	 * 
	 * @return bool 
	 */
	public function IsClientState()
	{
		return (isset($this->_event['clientstate']));
	}

	/**
	 * Gives back the Clientstate if it exists in the event-array
	 * 
	 * @return int 	clientstate 
	 * 				0 = the event does not belong to a client
	 * 				1 or higher = the location or the organizer is 
	 * 							registered as a client 
	 */
	public function ClientState()
	{
		return ($this->IsClientState())? $this->_event['clientstate'] : 0; 
	}	
	
	/**
	 * Is there a scene in the event-array?
	 * 
	 * @return bool 	 
	 */
	public function IsScene()
	{
		return (isset($this->_event['scene']));
	}
	
	/**
	 * Gives back the Scene of the event
	 * 
	 * @return string 	e.g. 'salsa', 'tango', 'mediation'... 	 
	 */
	public function Scene()
	{
		return ($this->IsScene())? $this->_event['scene'] : '';		 
	}

	
	/**
	 * Is there a category in the event-array?
	 * 
	 * @return bool 	 
	 */
	public function IsCategory()
	{
		return (isset($this->_event['category']));
	}

	/**
	 * Gives back the Category of the event
	 * 
	 * @return string 	e.g. 'party', 'concert', 'retreat'... 	 
	 */
	public function Category()
	{
		return ($this->IsCategory())? $this->_event['category'] : '';
	}	

	/**
	 * Is there a category-group in the event-array?
	 * 
	 * @return bool 	 
	 */
	public function IsCatgroup()
	{
		return (isset($this->_event['catgroup']));
	}
	
	/**
	 * Gives back the Category-group of the event
	 * 
	 * @return string 	one of each 'event', 'workshop', 'lesson' 	 
	 */
	public function Catgroup()
	{
		return ($this->IsCatgroup())? $this->_event['catgroup'] : '';		 
	}	
	
	/**
	 * Is there a start-value in the event-array?
	 * 
	 * @return bool 	 
	 */
	public function IsStart()
	{
		return (isset($this->_event['start']));
	}
	
	/**
	 * Get a formatted string of the events start date
	 *
	 * @param   string 	format: see PHP date function 
	 *
	 * @return  mixed	the formatted end date or the raw time	
	 */	
	public function Start( $format = '' )
	{
		if ( $this->IsStart() )
		{
			if ( empty($format) )
				return $this->start;
			else 
				return @date( $format, $this->start );
		}
		return 0;
	}
	
	/**
	 * Is there an end-value in the event-array?
	 * 
	 * @return bool 	 
	 */
	public function IsEnd()
	{
		return (isset($this->_event['end']));
	}
	
	/**
	 * Get a formatted string of the events end date
	 *
	 * @param   string 	format: see PHP date function 
	 *
	 * @return  mixed	the formatted end date or the raw time	
	 */	
	public function End( $format = '' )
	{
		if ( $this->IsEnd() )
		{
			if ( empty($format) )
				return $this->end;
			else 
				return @date( $format, $this->end );
		}
		return 0;
	}
	

	/**
	 * Is the requested location number valid?
	 * 
	 * @param 	int index of the location
	 * 
	 * @return bool 	 
	 */
	public function IsLocation( $index = 0 )
	{
		return (isset($this->_event['personlist']['location'][$index]));
	}
	

	/**
	 * Get an FastBroadcastPerson object of the Location
	 * 
	 * @param 	int 	index of the location. The default is zero.
	 * 
	 * @return 	mixed  	gives back an object of the class FastBroadcastPerson
	 * 					when the Location exists; otherwise null
	 */
	public function Location( $index = 0 )
	{
		$person = null;
		if ( $this->IsLocation( $index ) )
		{
			require_once dirname(__FILE__).'/FastBroadcastPerson.class.php';		
			$person = new FastBroadcastPerson( 
				$this->_event['personlist']['location'][$index] 
			);
		}
		return $person; 
	}

	/**
	 * Person involved
	 * 
	 * @param $id int or string with one or more ids
	 * 
	 * @return bool true if one of the given personids matches with
	 * 				the ids assigned to the event 
	 */
	public function PersonInvolved( $id )
	{
		$id = (string)$id;
		$involved = false;
		
		if ( isset($this->_event['personlist']) )
		{
			foreach ( $this->_event['personlist'] as $k => $list )
			{
				foreach ( $list as $person ) 
				{
					if ( isset($person['id']) && strpos( $id, $person['id'] ) !== false )
					{
						$involved = true;
						break;
					} 
				}
				if ( $involved )
					break;
			}
		}
		
		return $involved;
	}
	
	/**
	 * Is the requested organizer valid?
	 * 
	 * @param 	int index of the organizer
	 * 
	 * @return bool 	 
	 */
	public function IsOrganizer( $index = 0 )
	{
		return (isset($this->_event['personlist']['organizer'][$index]));
	}
	
	/**
	 * Get an FastBroadcastPerson object of the Organizer
	 * 
	 * @param 	int 	index of the organizer, The default is zero.
	 * 
	 * @return 	mixed  	gives back an object of the class FastBroadcastPerson
	 * 					when the Location exists; otherwise null
	 */
	public function Organizer( $index = 0 )
	{
		$person = null;
		if ( $this->IsOrganizer( $index ) )
		{
			require_once dirname(__FILE__).'/FastBroadcastPerson.class.php';			
			$person = new FastBroadcastPerson( 
				$this->_event['personlist']['organizer'][$index] 
			);
		}
		return $person; 
	}

	/**
	 * Is the requested guest valid?
	 * 
	 * @param 	int index of the guest
	 * 
	 * @return bool 	 
	 */
	public function IsGuest( $index = 0 )
	{
		return (isset($this->_event['personlist']['guest'][$index]));
	}
	
	/**
	 * Get an FastBroadcastPerson object of the Guest 
	 * 
	 * @param 	int 	index of the guest, The default is zero.
	 * 
	 * @return 	mixed  	gives back an object of the class FastBroadcastPerson
	 * 					when the Location exists; otherwise null
	 */
	public function Guest( $index = 0 )
	{
		$person = null;
		if ( $this->IsGuest( $index ) )
		{
			require_once dirname(__FILE__).'/FastBroadcastPerson.class.php';			
			$person = new FastBroadcastPerson( $this->_event['personlist']['guest'][$index] );
		}
		return $person; 
	}
	
	
	/**
	 * Is the requested dj valid?
	 * 
	 * @param 	int index of the dj
	 * 
	 * @return bool 	 
	 */
	public function IsDj( $index = 0 )
	{
		return (isset($this->_event['personlist']['dj'][$index]));
	}
	
	/**
	 * Get an FastBroadcastPerson object of the Dj 
	 * 
	 * @param 	int 	index of the dj, The default is zero.
	 * 
	 * @return 	mixed  	gives back an object of the class FastBroadcastPerson
	 * 					when the Location exists; otherwise null
	 */
	public function Dj( $index = 0 )
	{
		$person = null;
		if ( $this->IsDj( $index ) )
		{
			require_once dirname(__FILE__).'/FastBroadcastPerson.class.php';			
			$person = new FastBroadcastPerson( $this->_event['personlist']['dj'][$index] );
		}
		return $person; 
	}
	
	/**
	 * If the indexed Image exists this function will return the name of the image
	 * 
	 * @param	int	index
	 * 
	 * @return	bool
	 */	
	function ImageName( $index = 0 )
	{
		return ( $this->IsImage($index ))? $this->_event['images'][$index]['NAME'] : '';
	}
	

	/**
	 * The URL of the iCal presentation of the event
	 * 
	 * @return	string	the url 
	 */
	function ICalURL()
	{
		return	  FastBroadcast::$scheme
				. FastBroadcast::HOST
				. FastBroadcast::DOWNLOAD_PATH
				. 'rpc_method=V001_Calendar_GetICal&amp;'
				. 'rpc_accept=data&amp;'
				. 'eventid='. $this->ID() . '&amp;'
				. 'eventnr='. $this->Nr();
	}
		
	
	/**
	 * The URL of Files to Download
	 * 
	 * @param 	int		the index of the file in the event array
	 * 
	 * @return	string	the path, if an valid download-row is indexed,
	 * 					otherwise an empty string is returned 
	 */
	function DownloadURL( $index )
	{
		$url = '';
		if ( isset( $this->_event['downloads'][$index]) )
		{
			$url	= FastBroadcast::$scheme
					. FastBroadcast::HOST
					. FastBroadcast::DOWNLOAD_PATH
					. 'rpc_method=V001_Calendar_GetFile&amp;'
					. 'eventid='. $this->Id(). '&amp;'
					. 'file='. $this->_event['downloads'][$index]['NAME'];
		}
		return $url;
	}
	
	/**
	 * if a flyer exist in the imagetable this function will return true
	 * 
	 * @param	int 	index
	 * 
	 * @return	bool
	 */
	public function IsFlyer( &$index )
	{
		$isFlyer 	= false;
		$_index		= 0; 
		if ( isset( $this->_event['images'][0] ) )
		{
			foreach ( $this->_event['images'] as $image )
			{
				if ( $image['NAME'] == 'flyer_01' )
				{
					$isFlyer 	= true;
					$index 		= $_index;
					break;
				}
				$_index++;
			}
		}
		return $isFlyer;
	}
	
	
	/**
	 * The URL of an image identified by index 
	 * 
	 * @param	int		index
	 * @param	int		width oft the image
	 * @param	int		height of the image
	 * 
	 * @return	string	the url of the image if the index is valid
	 * 					otherwise an empty string
	 */
	function ImageURL( $index=0, $width = 0, $height = 0  )
	{
		$url = '';
		if ( $this->IsImage( $index ) )
		{
			$url	= FastBroadcast::$scheme
					. FastBroadcast::HOST
					. FastBroadcast::IMAGE_PATH
					. 'rpc_method=V001_Calendar_GetImage&amp;'
					. 'eventid='. $this->Id(). '&amp;'
					. 'file='. $this->ImageName($index);
	
			if ( $width > 0 )
				$url .= "&amp;width=$width";
			if ( $height > 0 )
				$url .= "&amp;height=$height";
		}
		return $url;				
	}
	
	/**
	 * a complete HTML img-tag is created and returned by this method
	 * the params are equal to the method ImageURL
	 * 
	 * @param	int		index
	 * @param	int		width
	 * @param	int		height
	 * @param	bool  	set_itemprop 
	 * 
	 * @return	string	the html img-tag
	 */
	function ImageTag( $index=0, $width = 0, $height = 0, $set_itemprop=false )
	{
		$tag = '';
		$url = $this->ImageURL($index, $width, $height );
		if ( !empty( $url ) )
		{
			$itemprop = ($set_itemprop)? $this->SchemaOrg()->image() : ''; 
		
			$width = ($width)? 'width="'.$width.'" ' : '';
			$tag = '<span class="img_shell"><img '. $itemprop . $width . 'src="' . $url . '" alt="" /></span>';
		} 
		return $tag; 
	}	
	/**
	 * Checks the existance of an image at pos $index
	 * 
	 * @param	int		index
	 * 
	 * @return 	bool
	 */	
	public function IsImage( $index = 0 )
	{
		return (isset($this->_event['images'][$index]));
	}
		
	/**
	 * Check for existens of a slogan-string in event-array.
	 *
	 * @return  bool	
	 */	
	public function IsSlogan()
	{
		return (isset( $this->_event['slogan']) && $this->_event['slogan'] );
	}

	/**
	 * Get the slogan of event-struct.
	 *
	 * @return  string	the slogan or an empty string	
	 */	
	public function Slogan()
	{
		return $this->Convert( ($this->IsSlogan())? $this->_event['slogan'] : '' ); 
	}	

	/**
	 * Check wether there is to display cost information in the template
	 *
	 * @return  bool	
	 */	
	public function CostBoxRequired()
	{
		$required = false;
		if ( $this->CostPerUnit() > 0 || $this->CostFree() > 0 )
			$required = true;
		
		if ( $this->CostVVK() > 0 || $this->CostAK() > 0 )
			$required = true;					

		if ( $this->CostText() || $this->CostText() )
			$required = true;					

		if ( $this->CostTable() || $this->CostTable() )
			$required = true;					
		
		return $required;
	}
	
	/**
	 * Check for existens of a valid cost row in event-array.
	 *
	 * @return  bool	
	 */	
	public function IsCost()
	{
		return (isset( $this->_event['cost']) && $this->_event['cost']);	
	}
	
	
	
	/**
	 * Check wether the event is for free
	 *
	 * @return  bool	
	 */	
	public function IsCostFree()
	{
		return (isset( $this->_event['cost']['FREE']) && $this->_event['cost']['FREE']);	
	}
	
	/**
	 * Check wether the event is for free
	 *
	 * @return  bool	
	 */	
	public function CostFree()
	{
		return ($this->IsCostFree())? $this->_event['cost']['FREE'] : '0';		
	}

	/**
	 * Is there a cost per unit entry in the event?
	 * 
	 * @return	bool
	 */
	public function IsCostPerUnit()
	{
		return (isset( $this->_event['cost']['PERUNIT']) && $this->_event['cost']['PERUNIT']);	
	}
	
	/**
	 * Give back the cost per unit value if it exists
	 * 
	 * @return string	the cost per unit 
	 */
	public function CostPerUnit()
	{
		return ($this->IsCostPerUnit())? $this->_event['cost']['PERUNIT'] : '';		
	}

	/**
	 * Is there a VVK entry in the event? (VVK means Vorverkauf)
	 * 
	 * @return	bool
	 */
	public function IsCostVVK()
	{
		return (isset( $this->_event['cost']['VVK']) && $this->_event['cost']['VVK'] );	
	}
	
	/**
	 *	Gives back the VVK (Vorverkauf) value if it exists in the event 
	 * 
	 * @return string
	 */
	public function CostVVK()
	{
		return ($this->IsCostVVK())? $this->_event['cost']['VVK'] : '';		
	}
	
	/**
	 * Is there a AK entry in the event? (AK means Abendkasse)
	 * 
	 * @return	bool
	 */
	public function IsCostAK()
	{
		return (isset( $this->_event['cost']['AK']) && $this->_event['cost']['AK'] );	
	}

	/**
	 *	Gives back the AK (Abendkasse) value if it exists in the event 
	 * 
	 * @return string
	 */
	public function CostAK()
	{
		return ($this->IsCostAK())? $this->_event['cost']['AK'] : '';		
	}

	/**
	 * Checks wether the textfield for the price for mor complex prices is used
	 * 
	 * @return	bool
	 */
	public function IsCostText()
	{
		return (isset( $this->_event['cost']['TEXT']) && $this->_event['cost']['TEXT']);	
	}
	/**
	 * Give back the text of the cost
	 * 
	 * @return	string
	 */
	public function CostText()
	{
		return $this->Convert( ($this->IsCostText())? $this->_event['cost']['TEXT'] : '' );		
	}

	/**
	 * Is the table used for the specification of the event price
	 * 
	 * @return	bool
	 */
	public function IsCostTable()
	{
		return (isset( $this->_event['cost']['costtable']) );	
	}
	
	/**
	 * the cost may be specified in a two column table
	 * 
	 * @return 	mixed	an array for the costtabele if it exists or null
	 */
	public function CostTable()
	{
		return ($this->IsCostTable())? $this->_event['cost']['costtable'] : null;		
	}

	/**
	 * A program may be specified as a table width two columns represented as an array
	 * by the user for an event.
	 * 
	 * @return bool
	 */
	public function IsProgram()
	{
		return (isset( $this->_event['program']) );	
	}
	
	/**
	 * A program may be specified as a table width two columns represented as an array
	 * by the user for an event.
	 * 
	 * @return	mixed	array if there is a program otherwise null
	 */
	public function Program()
	{
		return ($this->IsProgram())? $this->_event['program'] : null;		
	}

	/**
	 * does the event have something to download (php-file for example)
	 * 
	 * @return	bool
	 */
	public function IsDownload()
	{
		return (isset( $this->_event['downloads']) );	
	}
	
	/** 
	 * download are specified in an array with two columns, the text and the 
	 * file specification
	 * 
	 * @return	mixed	array, if there is something to download otherwise null
	 */
	public function Download()
	{
		return ($this->IsDownload())? $this->_event['downloads'] : null;		
	}
	
	/**
	 * this method can be used to resize the amount of text of the event description
	 * 
	 * @param	string	the text to resize
	 * @param	int		max lines oft the resized text
	 * @param	int		max char per line ( requred to braek paragraphs to lines)
	 * @param	string	for html output the eol should be set to a br tag
	 * @param	string	if the result is smaller when the input, an ellipse or an 
	 * 					other string is added to the end of the text
	 */
	function ResizeText( $text_in, $max_lines, $max_chars_per_line, $eol, $ellipse = ' ...' )
	{
		// erase empty lines at the beginning 
		$text_in = ltrim( $text_in );
		
		$text_in = str_replace( '\n\n','\n',$text_in );
		
		$lines = explode( '\n', $text_in );		
		$text_out = '';	
		
		$i = 0;
		foreach ( $lines as $line )
		{
			$tlines = strlen( $line ) / $max_chars_per_line + 1;
			if ( $i + $tlines >= $max_lines )
			{
				$chars = ($max_lines - $i) * $max_chars_per_line;
				$line = substr( $line, 0, $chars ); 
				$line = substr( $line, 0, strrpos($line,' ') );
				if ( $ellipse )
				 	$line .= $ellipse;
			}
			$text_out .= $line . $eol;
			$i += $tlines;
			if ( $i >= $max_lines )
				break;
		}
		return $text_out;
	}	

	/**
	 * 
	 * 
	 * @param 	string	$format
	 * @param	bool	$span 
	 * 
	 * @return 	string 
	 */
	public function RenderLine( $format, $span=true )
	{
		if ( empty($format) )
			return '';
		
		if ( strpos( $format, '$date') !== false )
		{
			$replace = $this->sprintf('{start_d}.{start_m}.');
			if ( $span )
				$replace = '<span class="date">' . $replace . '</span>';
			$format = str_replace( '$date', $replace, $format );
		}
		
		if ( strpos( $format, '$time') !== false )
		{
			$replace = $this->sprintf('{start_H}:{start_i}');
			if ( $span )
				$replace = '<span class="time">' . $replace . '</span>';
			$format = str_replace( '$time', $replace, $format );
		}
		
		if ( strpos( $format, '$city') !== false )
		{
			$replace = $this->sprintf('{location_city}');
			if ( $span )
				$replace = '<span class="city">' . $replace . '</span>';
			$format = str_replace( '$city', $replace, $format );
		}

		if ( strpos( $format, '$summary') !== false )
		{
			$replace = $this->Summary();
			if ( $span )
				$replace = '<span class="summary" ' . $this->SchemaOrg()->name() . '>' . $replace . '</span>';
			$format = str_replace( '$summary', $replace, $format );
		}
		
		if ( strpos( $format, '$location') !== false )
		{
			$replace = $this->sprintf('{location_name}');
			if ( $span )
				$replace = '<span class="location">' . $replace . '</span>';
			$format = str_replace( '$location', $replace, $format );
		}
		if ( strpos( $format, '$street') !== false )
		{
			$replace = $this->sprintf('{location_street}');
			if ( $span )
				$replace = '<span class="street">' . $replace . '</span>';
			$format = str_replace( '$street', $replace, $format );
		}
		if ( strpos( $format, '$zip') !== false )
		{
			$replace = $this->sprintf('{location_zip}');
			if ( $span )
				$replace = '<span class="zip">' . $replace . '</span>';
			$format = str_replace( '$zip', $replace, $format );
		}
		
		
		if ( strpos( $format, '$slogan') !== false )
		{
			$replace = $this->Slogan();
			if ( !empty($replace) && $span )
				$replace = '<span class="slogan">' . $replace . '</span>';
			$format = str_replace( '$slogan', $replace, $format );
		}
		
		return $format;
		
/*	
		$items = explode( ' ', $format );
		$line = '';
		if ( !empty($items))
		{
			foreach ( $items as $item )
			{
				switch ( $item )
				{
				case 'date':
					$line .= '<span class="date">' . $this->sprintf('{start_d}.{start_m}.') . '</span>'; 		
					break;
				case 'time':
					$line .= '<span class="time">' . $this->sprintf('{start_h}:{start_i}.') . '</span>';					
					break;
				case 'city':
					$line .= '<span class="city">' . $this->sprintf('{location_city}') . '</span>';					
					break;
				case 'summary':
					$line .= '<span class="summary">' . $this->Summary() . '</span>';					
					break;
				case 'slogan':
					$line .= '<span class="slogan">' . $this->Slogan() . '</span>';					
					break;
				}
			}
		}
		return $line;		
*/		

	}
	
	/**
	 * complex formattings can be done with this method. Variables must be enclosed
	 * with brackets {variable}. 
	 * sample: "{location_city}: {start_H}:{start_i} {location_fullname}"
	 * for the resolution of time-variables see the date funktion in php.net 
	 * 
	 * @param	string	format
	 * 
	 * @return	the formatstring where the valid params are resolved
	 */
	public function sprintf( $format )
	{
		$pattern = "/{[^}]*}/";
		preg_match_all($pattern, $format, $matches);
		
		$location = $this->Location();
		$organizer = $this->Organizer();		
			
		foreach ( $matches[0] as $search )
		{
			$replace 	= '';
			$var		= trim( $search, '{}' );
			$var_parts 	= explode( '_', $var );
			
			if ( count( $var_parts ) > 1 )
			{
				switch ( $var_parts[0] )
				{
				case 'location':
					if ( is_object( $location ) )
						$replace = $location->get( $var_parts[1] );
					break;
				case 'organizer':
					if ( is_object( $organizer ) )
						$replace = $organizer->get( $var_parts[1] );
					break;
				case 'start':
					switch ( $var_parts[1] )
					{
					case 'D':
					case 'l':
					case 'F':
						$replace = FastBroadcast::Translate( $this->Start( $var_parts[1] ) );
						break;
						//else weiter
					default:
						$replace = $this->Start( $var_parts[1] );
					}
					break;
				case 'end':
					switch ( $var_parts[1] )
					{
					case 'D':
					case 'l':
					case 'F':
						$replace = FastBroadcast::Translate( $this->End( $var_parts[1] ) );
						break;
						//else weiter
					default:
						$replace = $this->End( $var_parts[1] );
					}
					break;
				}
			}
			else
			{
//				switch ( $var )
//				{
//				}
			}
			$format = str_replace( $search, $replace, $format );			
		}
		
		return $format;
	}
	
	
	/**
	 * Check for existens of a event-Description
	 *
	 * @return  bool	
	 */	
	public function IsDescription()
	{
		return (isset( $this->_event['description']) && $this->_event['description'])? true : false;
	}
	
	/**
	 * Get the description of event-struct. If the values for lines and columns are 
	 * specified the method will use the ResiseTest Method to resize the text
	 *
	 * @param   int		columns for resizing
	 * @param   int		lines for resizing*
	 * @param	string	the linebreakes usually will be replaced with a br tag
	 *
	 * @return  string	the description or an empty string resized by the params	
	 */	
	public function Description( $lines = 0, $columns = 0, $eol = '<br />')
	{
		$description = '';
		if ( $this->IsDescription() )
		{
			if ( $columns && $lines )
				$description = $this->ResizeText( $this->Convert( $this->_event['description']), $lines, $columns, $eol );
			else 
			{
				$description = $this->Convert( $this->_event['description'] );
				if ( $this->ClientState() > 0 )
					$description = $this->URLsToLinks( $description );
					
				$description = str_replace( "\\n", $eol, $description );
			}
		}
		return $description;
	}
	
	/**
	 * Converts URLs in a given String to HTML-a-tags
	 * 
	 * @param	string	The text with URLs
	 * 
	 * @return 	string  The URLs are translates to HTML a tags  
	 */
	public function URLsToLinks( $text )
	{
		preg_match_all('/(http:\/\/[^\s]+)/', $text, $href_list);
		if ( !empty( $href_list ) )
		{
			foreach ( $href_list[0] as $href )
			{
				// if there is an eol-marker erase it
				$href		= rtrim( str_replace( "\\n", ' ', $href ) );
				$offset 	= (strncmp( $href, 'http://www.', 11 ))? 7 : 11;
				$link_text 	= substr( $href, $offset );
		 
				$replace = '<a href="' . $href. '">'. $link_text . '</a>'; 	
			
				$text = str_replace( $href, $replace, $text );
			}
		}
		
		return $text;
	}
	
}
